<?php

namespace YOOtheme\Widgetkit\Content\Facebook;

use YOOtheme\Widgetkit\Framework\Application;
use YOOtheme\Widgetkit\Framework\ApplicationAware;

define('FACEBOOK_SDK_V4_SRC_DIR', __DIR__ . '/Facebook/');
require_once __DIR__ . '/Facebook/autoload.php';
use Facebook\Facebook;

/**
 * Instagram API class
 *
 * API Documentation: http://instagram.com/developer/
 * Class Documentation: https://github.com/cosenary/Instagram-PHP-API
 *
 * @author Christian Metz
 * @since 30.10.2011
 * @copyright Christian Metz - MetzWeb Networks 2011-2014
 * @version 2.2
 * @license BSD http://www.opensource.org/licenses/bsd-license.php
 */
class FacebookOAuth extends ApplicationAware
{
    /**
     * Default constructor.
     *
     * @param array|string $config Instagram configuration data
     *
     * @return void
     *
     * @throws \MetzWeb\Instagram\InstagramException
     */
    public function __construct(Application $app, $config)
    {
		$this->app = $app;
    }
	
	public function connect($app_id = '', $app_secret = '')
	{
		$default_access_token 	= $app_id.'|'.$app_secret;
		
		$fb = new Facebook(array(
			'app_id' => $app_id,
			'app_secret' => $app_secret,
			'default_access_token' => $default_access_token,
			'default_graph_version' => 'v2.4'
		));
		
		// Sets the default fallback access token so we don't have to pass it to each request
		try {
			$token = $fb->get('/oauth/access_token?client_id='.$app_id.'&client_secret='.$app_secret.'&grant_type=client_credentials')->getGraphNode();
			$fb->setDefaultAccessToken($token->getField('access_token'));
		} catch(Facebook\Exceptions\FacebookResponseException $e) {
			// When Graph returns an error
			echo 'Graph returned an error: ' . $e->getMessage();
			exit;
		} catch(Facebook\Exceptions\FacebookSDKException $e) {
			// When validation fails or other local issues
			echo 'Facebook SDK returned an error: ' . $e->getMessage();
			exit;
		}
		
		return $fb;
	}

	
	/**
     * Issues the API.
     *
     * @param $method
     * @param array $params
     * @param array $key
     * @return array|mixed
     * @throws \Exception
     */
    public function get($method, $params = array(), $key = array())
    {
		if (!isset($key['app_id']) || !$key['app_id'] || !isset($key['app_secret']) || !$key['app_secret']) {
            throw new \Exception('Please check your Facebook Settings.');
        }
		
		$fb = $this->connect($key['app_id'], $key['app_secret']);
					
        if ($method === 'feed') {
			
			try {
				$q = 'feed?fields=created_time,story,message,from,to,picture,full_picture,object_id,source,status_type,name,description,likes,type,link';
				$response = $fb->get('/'.$params['search'].'/'.$q.'&limit='.$params['limit']);
				$items = $response->getGraphEdge();
			} catch(Facebook\Exceptions\FacebookResponseException $e) {
				// When Graph returns an error
				echo 'Graph returned an error: ' . $e->getMessage();
				exit;
			} catch(Facebook\Exceptions\FacebookSDKException $e) {
				// When validation fails or other local issues
				echo 'Facebook SDK returned an error: ' . $e->getMessage();
				exit;
			}

        } else {
            throw new \Exception('Unknown API method');
        }
		
        // object -> array recursive
        //$items = json_decode(json_encode($items), true);
		
        return $items;

    }
		
    /**
     * Fetches tweets from cache or API.
     *
     * @param $method
     * @param $key
     * @param $params
     * @param $content
     * @return array|mixed
     * @throws \Exception
     */
    public function fetch($method, $key, $params, $content)
    {
		$token = $key;
		
        // Cache settings
        $now = time();
        $expires = 5 * 60;
		
		$items = array();
		
        // Cache invalid?
        if (array_key_exists('hash', $content) // never cached
            || $now - $content['hashed'] > $expires // cached values too old
            || $this->hash($method, $params, $token) != $content['hash'] // content settings have changed
        ) {
			
            try {
                $response = $this->get($method, $params, $token);
				
				$count = 0;
                foreach ($response as $item) {
					
                    $fb_item = $this->prepare($item, $content);

                    // check maximum tweet count
                    if ($count >= $content['limit']) {
                        continue;
                    }
                    
                    $items[] = $fb_item;
                    $count++;
                }

                // write cache
                $content['prepared'] = json_encode($items);
                $content['hash'] = $this->hash($method, $params, $token);
                $content['hashed'] = $now;
                unset($content['error']);

                $this->app['content']->save($content->toArray());

                return $items;
            } catch (\Exception $e) {
                // Fallback to cache and log of API error
                $content['error'] = $e->getMessage();
                $this->app['content']->save($content->toArray());
            }
        }
		
        // read from cache
        $items = json_decode($content['prepared'], true);
		
        return $items ? : array();
    }

    /**
     * Prepares tweets for displaying.
     *
     * @param $tweet
     * @param $content
     * @return array
     */
    public function prepare($fb_item, $content)
    {
		//$utc = new \DateTimeZone('UTC');
		//$created = \DateTime::createFromFormat('U', $fb_item['created_time'], $utc);
		//$fb_item['created_time']->format('d-m-Y H:i:s');
					
		$item = array(
            'title' => $this->escape($fb_item['from']['name']),
			'content' => $this->escape($fb_item['message']),
			'date' => ($content['show_date']) ? $fb_item['created_time']->format('d-m-Y H:i:s') : '',
            'link' => $fb_item['link'],
            'location' => null,
            'media' => '',
			'fb_type' => $fb_item['type'],
			'facebook_item' => $fb_item
        );
		
		if (isset($fb_item['object_id'])) {
            $item['media'] = "https://graph.facebook.com/".$fb_item['object_id']."/picture/?type=".$content['media'];
        }
		
		if (isset($fb_item['full_picture'])) {
            $item['media'] = $fb_item['full_picture'];
        }
		
		$item['content'] = nl2br( \JHtml::_('string.truncateComplex', $this->escape($fb_item['message']), $content['limit_content']) );
		
        return $item;
    }

    /**
     * @param $str
     * @return string
     */
    protected function escape($str)
    {
        return htmlspecialchars($str, ENT_COMPAT, 'UTF-8', false);
    }

    /**
     * Hashes request parameters.
     *
     * @param $method
     * @param $token
     * @param $params
     * @return string
     */
    protected function hash($method, $token, $params)
    {
        $fields = array($method, $token, $params);

        return md5(serialize($fields));
    }
}
